/** @file   statistics.cpp
 * @brief   Implementation of Statistics - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#include "statistics.h"
#include <string>
using std::istream;
using std::ostream;
using std::endl;
using std::string;

namespace WeWantWar {

///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
Statistics::Statistics() :
  m_score( 0 ),
  m_levelTimer( 0 )
{
  for ( int i=0; i < NUM_OF_ITEMS; i++ ) {
    m_levelstats[i] = 0;
    m_overallstats[i] = 0;
  }
}



/** Destructor
 */
Statistics::~Statistics()
{
}

/** Reads the statistics
 */
int Statistics::readFrom( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</wewantwar_statistics>" ) {
      break;
      
    } else if ( tmp == "score:" ) {
      rIn >> m_score;
      
    } else if ( tmp == "shots:" ) {
      rIn >> m_overallstats[ SHOTS ];
      
    } else if ( tmp == "hits:" ) {
      rIn >> m_overallstats[ HITS ];
      
    } else if ( tmp == "kills:" ) {
      rIn >> m_overallstats[ KILLS ];
      
    }
  }
  return 0;
}



/** Saves the statistics
 */
int Statistics::writeTo( ostream& rOut ) const
{
  rOut << "<wewantwar_statistics>" << endl;
  rOut << "    " << "score: " << m_score << endl;
  rOut << "    " << "shots: " << this->getOverallStats( SHOTS ) << endl;
  rOut << "    " << "hits: " << this->getOverallStats( HITS ) << endl;
  rOut << "    " << "kills: " << this->getOverallStats( KILLS ) << endl;
  rOut << "</wewantwar_statistics>" << endl;
  return 0;
}



/** Resets the level statistics
 */
void Statistics::resetLevelStats()
{
  m_levelTimer = 0;
  for ( int i=0; i < NUM_OF_ITEMS; i++ ) {
    m_levelstats[i] = 0;
  }
}



/** Resets all the statistics
 */
void Statistics::resetAllStats()
{
  m_levelTimer = 0;
  m_score = 0;
  for ( int i=0; i < NUM_OF_ITEMS; i++ ) {
    m_levelstats[i] = 0;
    m_overallstats[i] = 0;
  }
}



/** Updates the level timer.
 */
void Statistics::updateLevelTimer()
{
  m_levelTimer += 1;
}



/** Updates the statistics by given event
 */
void Statistics::updateStatistics( StatType stat, ObjectID::Type oid )
{
  switch ( stat ) {
    case ( SHOTS ): {
      m_levelstats[ SHOTS ] += 1;
      m_overallstats[ SHOTS ] += 1;
      break;
    }
    case ( HITS ): {
      m_levelstats[ HITS ] += 1;
      m_overallstats[ HITS ] += 1;
      break;
    }
    case ( KILLS ): {
      m_levelstats[ KILLS ] += 1;
      m_overallstats[ KILLS ] += 1;
      this->grantKillingScore( oid );
      break;
    }
  }
}



/** Grants given amount of scores
 */
void Statistics::grantScores( int scores )
{
  m_score += scores;
}



///
/// Getter methods
/// ==============

/** Returns the total score
 */
int Statistics::score() const
{
  return m_score;
}



/** Returns the statistics of current level.
 */
int Statistics::getLevelStats( StatType stat ) const
{
  return m_levelstats[ stat ];
}



/** Returns the overall statistics
 */
int Statistics::getOverallStats( StatType stat ) const
{
  return m_overallstats[ stat ];
}



/** Returns the level timer
 */
int Statistics::levelTimer() const
{
  return m_levelTimer;
}





///
/// Private or Protected methods
/// ============================

/** Grants the killing scores
 */
void Statistics::grantKillingScore( ObjectID::Type oid )
{
  switch ( oid ) {
    case ( ObjectID::TYPE_CARNIVOREALIEN ): {
      m_score += 200;
      break;
    }
    case ( ObjectID::TYPE_SMALLWORMALIEN ): {
      m_score += 120;
      break;
    }
    case ( ObjectID::TYPE_WINGEDALIEN ): {
      m_score += 220;
      break;
    }
    case ( ObjectID::TYPE_PROCTORALIEN ): {
      m_score += 450;
      break;
    }
    case ( ObjectID::TYPE_MINIGUNALIEN ): {
      m_score += 500;
      break;
    }
    case ( ObjectID::TYPE_PREDATORALIEN ): {
      m_score += 350;
      break;
    }
    case ( ObjectID::TYPE_LIGHTBALLALIEN ):
    case ( ObjectID::TYPE_FLAMERALIEN ):
    case ( ObjectID::TYPE_SENTRYGUN ): {
      m_score += 800;
      break;
    }
    case ( ObjectID::TYPE_CIVILIAN ): {
      m_score -= 1000;
      break;
    }
    case ( ObjectID::TYPE_CAR ): {
      m_score -= 1000;
      break;
    }
    case ( ObjectID::TYPE_FIGHTER ): {
      m_score += 1200;
      break;
    }
  }
  if ( m_score < 0 ) {
    m_score = 0;
  }
}


} // end of namespace
